/*
 * ALSA driver for Panasonic UniPhier series.
 * 
 * Copyright (c) 2013 Panasonic corporation.
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 */

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/init.h>

#include "mn2ws-pcm.h"

MODULE_AUTHOR("Katsuhiro Suzuki <suzuki.katsuhiro002@jp.panasonic.com>");
MODULE_DESCRIPTION("Panasonic UniPhier MN2WS0220 PCM Driver");
MODULE_LICENSE("GPL");
MODULE_SUPPORTED_DEVICE(
	"{{Panasonic, MN2WS0220}, "
	"{Panasonic, MN2WS0220x}}");


#define MN2WS_PCM_DEV_NAME       "mn2ws0220-pcm"
#define MN2WS_PCM_DEV_COUNT      2

#define MN2WS_RES_PCM            0
#define MN2WS_RES_AIO2           1

#define MN2WS_PCM_REG_START      (0x5e00c000)
#define MN2WS_PCM_REG_SIZE       (0x00004000)
#define MN2WS_AIO2_REG_START     (0x5e006000)
#define MN2WS_AIO2_REG_SIZE      (0x00002000)

#define MN2WS_PCM_HW_BUF_SIZE    (PAGE_SIZE * 4)


//d250, d254, d258
#define PCM_PCMOCTL_0         (0x5e00d250)
#define PCM_PCMOCTL_1         (0x5e00d254)
#define PCM_IECOCTL           (0x5e00d258)

//f00c, f30c, fb0c
#define PCM_AUDCTL0(ch) \
	((ch == 0) ? 0x5e00f00c : \
	(ch == 1) ? 0x5e00f30c : \
	(ch == 2) ? 0x5e00fb0c : \
	0)

//f2a4, f5a4
#define PCM_EXTPCMPLAYCTL(ch) \
	((ch == 0) ? 0x5e00f2a4 : \
	(ch == 1) ? 0x5e00f5a4 : \
	0)
#define PCM_EXTPCMBUFSTADDR(ch) \
	((ch == 0) ? 0x5e00f2a8 : \
	(ch == 1) ? 0x5e00f5a8 : \
	0)
#define PCM_EXTPCMBUFENDADDR(ch) \
	((ch == 0) ? 0x5e00f2ac : \
	(ch == 1) ? 0x5e00f5ac : \
	0)

#define PLAYCTL_PLAYSW      ( 1 <<  0) //1'b1
#define PLAYCTL_FSOUTSEL    ( 1 <<  3) //1'b1
#define PLAYCTL_FS          ( 7 <<  4) //3'b111
#define PLAYCTL_SCALE       (15 <<  8) //4'b1111
#define PLAYCTL_DECSCALE    (15 << 16) //4'b1111


//f2b0, f5b0
#define PCM_EXTPCMBUFTHSIZE(ch) \
	((ch == 0) ? 0x5e00f2b0 : \
	(ch == 1) ? 0x5e00f5b0 : \
	0)
#define PCM_EXTPCMBUFWRPTR(ch) \
	((ch == 0) ? 0x5e00f2b4 : \
	(ch == 1) ? 0x5e00f5b4 : \
	0)
#define PCM_EXTPCMBUFRDPTR(ch) \
	((ch == 0) ? 0x5e00f2b8 : \
	(ch == 1) ? 0x5e00f5b8 : \
	0)


//Path: 
//  PCMIN1 --> (AudioIn0) --> (input-sel) --> PBin2
//  PBin2  --(Subsystem 07)--> Ringbuf 07(DDR)
#define PCMIN1         0
#define MAIN_IN        7

//AIN(PCMMINn)
#define MN2WS_A2PCMiNCtr1(n)             (0x5e006000 + 0x40 * (n))
#define MN2WS_A2PCMiNCtr2(n)             (0x5e006004 + 0x40 * (n))
#define MN2WS_A2PCMiNACCtr(n)            (0x5e006008 + 0x40 * (n))
#define MN2WS_A2PCMiNCounter(n)          (0x5e006010 + 0x40 * (n))
#define MN2WS_A2PCMiNRstCtr(n)           (0x5e00603c + 0x40 * (n))

//AIN(selector)
#define MN2WS_A2InAInIFCtrPBin0          (0x5e006530)
#define MN2WS_A2InAInIFCtrPBin1          (0x5e006534)
#define MN2WS_A2InAInIFCtrPBin2          (0x5e006538)
#define MN2WS_A2InAInIFCtrPBin3          (0x5e00653c)
#define MN2WS_A2InAInIFCtrPBin4          (0x5e006540)
#define MN2WS_A2InAInIFCtrPBin5          (0x5e006544)
#define MN2WS_A2InAInIFCtrPBin6          (0x5e006548)
#define MN2WS_A2InAInIFCtrPBin7          (0x5e00654c)

//A2D(subsystem)
#define MN2WS_CDA2D_STRT0                (0x5e007000)
#define MN2WS_CDA2D_PERFCNFG             (0x5e007008)
#define MN2WS_CDA2D_STAT0                (0x5e007020)
#define MN2WS_CDA2D_CHnCTRL1(n)          (0x5e007100 + 0x40 * (n))
#define MN2WS_CDA2D_CHnCTRL2(n)          (0x5e007104 + 0x40 * (n))
#define MN2WS_CDA2D_CHnSTAT(n)           (0x5e007110 + 0x40 * (n))
#define MN2WS_CDA2D_CHnIR(n)             (0x5e007114 + 0x40 * (n))
#define MN2WS_CDA2D_CHnIE(n)             (0x5e007118 + 0x40 * (n))
#define MN2WS_CDA2D_CHnID(n)             (0x5e00711c + 0x40 * (n))
#define MN2WS_CDA2D_CHnSRCAMODE(n)       (0x5e007120 + 0x40 * (n))
#define MN2WS_CDA2D_CHnDSTAMODE(n)       (0x5e007124 + 0x40 * (n))
#define MN2WS_CDA2D_CHnSRCSTRTADRS(n)    (0x5e007128 + 0x40 * (n))
#define MN2WS_CDA2D_CHnDSTSTRTADRS(n)    (0x5e00712c + 0x40 * (n))
#define MN2WS_CDA2D_CHnSIZE(n)           (0x5e007130 + 0x40 * (n))

//A2D(ring buffer)
#define MN2WS_CDA2D_RBFLUSH0             (0x5e007040)
#define MN2WS_CDA2D_PARTRESET0           (0x5e007050)
#define MN2WS_CDA2D_PARTRESET1           (0x5e007054)
#define MN2WS_CDA2D_RBnBGNADRS(n)        (0x5e007600 + 0x40 * (n))
#define MN2WS_CDA2D_RBnENDADRS(n)        (0x5e007604 + 0x40 * (n))
#define MN2WS_CDA2D_RBnBTH(n)            (0x5e007608 + 0x40 * (n))
#define MN2WS_CDA2D_RBnRTH(n)            (0x5e00760c + 0x40 * (n))
#define MN2WS_CDA2D_RBnSTAT(n)           (0x5e007610 + 0x40 * (n))
#define MN2WS_CDA2D_RBnIR(n)             (0x5e007614 + 0x40 * (n))
#define MN2WS_CDA2D_RBnIE(n)             (0x5e007618 + 0x40 * (n))
#define MN2WS_CDA2D_RBnID(n)             (0x5e00761c + 0x40 * (n))
#define MN2WS_CDA2D_RBnRDPTR(n)          (0x5e007620 + 0x40 * (n))
#define MN2WS_CDA2D_RBnWRPTR(n)          (0x5e007628 + 0x40 * (n))
#define MN2WS_CDA2D_RBnCNFG(n)           (0x5e007630 + 0x40 * (n))
#define MN2WS_CDA2D_RBnPTRCTRL(n)        (0x5e007634 + 0x40 * (n))


static int mn2ws0220_pcm_hwdep_force_aout(struct mn2ws_pcm_dev *dev);

static int mn2ws0220_pcm_play_setup(struct mn2ws_pcm_dev *dev);
static int mn2ws0220_pcm_play_mute(struct mn2ws_pcm_dev *dev);
static int mn2ws0220_pcm_play_start(struct mn2ws_pcm_dev *dev);
static int mn2ws0220_pcm_play_stop(struct mn2ws_pcm_dev *dev);
static unsigned long mn2ws0220_pcm_play_get_hwptr(struct mn2ws_pcm_dev *dev);
static void mn2ws0220_pcm_play_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr);
static int mn2ws0220_pcm_cap_setup(struct mn2ws_pcm_dev *dev);
static int mn2ws0220_pcm_cap_start(struct mn2ws_pcm_dev *dev);
static int mn2ws0220_pcm_cap_stop(struct mn2ws_pcm_dev *dev);
static unsigned long mn2ws0220_pcm_cap_get_hwptr(struct mn2ws_pcm_dev *dev);
static void mn2ws0220_pcm_cap_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr);
static int mn2ws0220_pcm_init_module(void);
static void mn2ws0220_pcm_exit_module(void);


static struct resource mn2ws_resources[] = {
	[MN2WS_RES_PCM] = {
		.start = MN2WS_PCM_REG_START, 
		.end   = MN2WS_PCM_REG_START + MN2WS_PCM_REG_SIZE, 
		.flags = IORESOURCE_MEM, 
	}, 
	[MN2WS_RES_AIO2] = {
		.start = MN2WS_AIO2_REG_START, 
		.end   = MN2WS_AIO2_REG_START + MN2WS_AIO2_REG_SIZE, 
		.flags = IORESOURCE_MEM, 
	}, 
};

struct mn2ws_pcm_desc mn2ws_pcm[] = {
	//ch 0
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.hwd = {
			.enabled       = 1, 
			.force_aout    = mn2ws0220_pcm_hwdep_force_aout, 
		}, 
		.play = {
			.enabled       = 1, 
			//0 means allocate buffer from the kernel
			.phys_addr_buf = 0, 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE, 
			//0 means use default size(= size_buf)
			//18.6ms(min: 1536bytes = 8ms)
			.size_use_fix  = 256 * 14, 
			.init          = NULL, 
			.term          = NULL, 
			.hardware      = mn2ws_pcm_pcmif_hardware_48k_16b_2ch, 
			.setup         = mn2ws0220_pcm_play_setup, 
			.mute          = mn2ws0220_pcm_play_mute, 
			.start         = mn2ws0220_pcm_play_start, 
			.stop          = mn2ws0220_pcm_play_stop, 
			.get_hwptr     = mn2ws0220_pcm_play_get_hwptr, 
			.set_devptr    = mn2ws0220_pcm_play_set_devptr, 
			.wait_hwevent  = mn2ws_pcm_pcmif_hrtimeout, 
			.copy          = mn2ws_pcm_play_copy_to_hw, 
			.silence       = mn2ws_pcm_play_silence_to_hw, 
		}, 
		.cap = {
			.enabled       = 1, 
			//0 means allocate buffer from the kernel
			.phys_addr_buf = 0, 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE, 
			//0 means use default size(= size_buf)
			//18.6ms(min: 1024bytes = 2.67ms)
			.size_use_fix  = 256 * 28, 
			.init          = NULL, 
			.term          = NULL, 
			.hardware      = mn2ws_pcm_pcmif_hardware_48k_32b_2ch, 
			.setup         = mn2ws0220_pcm_cap_setup, 
			.mute          = NULL, 
			.start         = mn2ws0220_pcm_cap_start, 
			.stop          = mn2ws0220_pcm_cap_stop, 
			.get_hwptr     = mn2ws0220_pcm_cap_get_hwptr, 
			.set_devptr    = mn2ws0220_pcm_cap_set_devptr, 
			.wait_hwevent  = mn2ws_pcm_pcmif_hrtimeout, 
			.copy          = mn2ws_pcm_cap_copy_to_alsa, 
			.silence       = mn2ws_pcm_cap_silence_to_alsa, 
		}, 
	}, 
	
	//ch 1
	{
		.res_regs      = mn2ws_resources, 
		.n_res_regs    = ARRAY_SIZE(mn2ws_resources), 
		
		.hwd = {
			.enabled       = 1, 
		}, 
		.play = {
			.enabled       = 1, 
			//0 means allocate buffer from the kernel
			.phys_addr_buf = 0, 
			.size_buf      = MN2WS_PCM_HW_BUF_SIZE, 
			//0 means use default size(= size_buf)
			//18.6ms(min: 1536bytes = 8ms)
			.size_use_fix  = 256 * 14, 
			.init          = NULL, 
			.term          = NULL, 
			.hardware      = mn2ws_pcm_pcmif_hardware_48k_16b_2ch, 
			.setup         = mn2ws0220_pcm_play_setup, 
			.mute          = mn2ws0220_pcm_play_mute, 
			.start         = mn2ws0220_pcm_play_start, 
			.stop          = mn2ws0220_pcm_play_stop, 
			.get_hwptr     = mn2ws0220_pcm_play_get_hwptr, 
			.set_devptr    = mn2ws0220_pcm_play_set_devptr, 
			.wait_hwevent  = mn2ws_pcm_pcmif_hrtimeout, 
			.copy          = mn2ws_pcm_play_copy_to_hw, 
			.silence       = mn2ws_pcm_play_silence_to_hw, 
		}, 
	}, 
};


const char *get_pcm_dev_name(void)
{
	return MN2WS_PCM_DEV_NAME;
}

int get_pcm_dev_count(void)
{
	return MN2WS_PCM_DEV_COUNT;
}

static int mn2ws0220_pcm_hwdep_force_aout(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	
	//settings: Main:DEC0, Sub:DEC1, IEC:DEC1
	
	//0    : enable : off
	pcm_writel(mmp, 0x0000c020, PCM_PCMOCTL_0);
	pcm_writel(mmp, 0x0001c028, PCM_PCMOCTL_1);
	pcm_writel(mmp, 0x00100008, PCM_IECOCTL);
	
	//0    : enable : on
	pcm_writel(mmp, 0x0000c021, PCM_PCMOCTL_0);
	pcm_writel(mmp, 0x0001c029, PCM_PCMOCTL_1);
	pcm_writel(mmp, 0x00100009, PCM_IECOCTL);
	
	//20-23: init fs    : 44.1kHz
	//18   : IEC copy   : can copy
	//17   : IEC output : PCM
	pcm_writel(mmp, 0x00040000, PCM_AUDCTL0(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0220_pcm_play_setup(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 st, ed;
	u32 v;
	
	//DMA buffer address
	st = dev->play.paddr_buf - 0x80000000;
	ed = st + dev->play.size_buf_use;
	pcm_writel(mmp, st, PCM_EXTPCMBUFSTADDR(dev->ch));
	pcm_writel(mmp, ed, PCM_EXTPCMBUFENDADDR(dev->ch));
	pcm_writel(mmp, st, PCM_EXTPCMBUFWRPTR(dev->ch));
	
	//DMA start immediately
	pcm_writel(mmp, 8, PCM_EXTPCMBUFTHSIZE(dev->ch));
	
	//Initialize the sound mode
	//16-19: Scaler decoder     : not set
	// 8-11: Scaler LPCM        : not set
	// 4- 6: 48kHz 2ch
	// 3   : Mix decoder output : not set
	// 0   : Start DMA          : not set
	v = 0;
	//v |= PLAYCTL_PLAYSW;
	//v |= PLAYCTL_FSOUTSEL;
	v |= PLAYCTL_FS;
	pcm_writel(mmp, v, PCM_EXTPCMPLAYCTL(dev->ch));
	
	//Set scaler
	mn2ws0220_pcm_play_mute(dev);
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0220_pcm_play_mute(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	//Set the scaler mode
	//16-19: Scaler decoder
	// 8-11: Scaler LPCM
	v = pcm_readl(mmp, PCM_EXTPCMPLAYCTL(dev->ch));
	v &= ~PLAYCTL_DECSCALE;
	v |= (dev->hwd.scale_dec << 16);
	v &= ~PLAYCTL_SCALE;
	v |= (dev->hwd.scale_pcm <<  8);
	pcm_writel(mmp, v, PCM_EXTPCMPLAYCTL(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0220_pcm_play_start(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	DPRINTF("start the playback DMA %d.\n", dev->ch);
	
	//start the DMA
	v = pcm_readl(mmp, PCM_EXTPCMPLAYCTL(dev->ch));
	v |= PLAYCTL_PLAYSW;
	pcm_writel(mmp, v, PCM_EXTPCMPLAYCTL(dev->ch));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0220_pcm_play_stop(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	DPRINTF("stop the playback DMA %d.\n", dev->ch);
	
	//stop the DMA
	v = pcm_readl(mmp, PCM_EXTPCMPLAYCTL(dev->ch));
	v &= ~PLAYCTL_PLAYSW;
	pcm_writel(mmp, v, PCM_EXTPCMPLAYCTL(dev->ch));
	
	return 0;
}

/**
 * PCM  HW ɤ߼äƤХåեˤĤơ
 * ߤ HW ɤ߼֤롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @return HW θߤɤ߼
 */
static unsigned long mn2ws0220_pcm_play_get_hwptr(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 v;
	
	v = pcm_readl(mmp, PCM_EXTPCMBUFRDPTR(dev->ch)) - 
		pcm_readl(mmp, PCM_EXTPCMBUFSTADDR(dev->ch));
	
	return v;
}

/**
 * PCM  HW ɤ߼äƤХåեˤĤơ
 * ߤΥǥХν񤭹֤߰ꤹ롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @param ptr ǥХθߤν񤭹߰
 */
static void mn2ws0220_pcm_play_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr)
{
	struct mem_mapping *mmp = &dev->map_regs[MN2WS_RES_PCM];
	u32 wr, wr_new;
	
	wr = pcm_readl(mmp, PCM_EXTPCMBUFWRPTR(dev->ch));
	wr_new = pcm_readl(mmp, PCM_EXTPCMBUFSTADDR(dev->ch)) + ptr;
	
	if (wr != wr_new) {
		pcm_writel(mmp, wr_new, PCM_EXTPCMBUFWRPTR(dev->ch));
	}
}




/**
 * ꡼׶ػ
 */
static int mn2ws0220_pcm_cap_setup(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mma = &dev->map_regs[MN2WS_RES_AIO2];
	u32 st, ed;
	
	st = dev->cap.paddr_buf;
	ed = st + dev->cap.size_buf_use;
	
	//----- reset -----
	//AIN(PCMIN1)
	// 7   : reset PCMIN   : assert
	// 0   : counter reset : on
	pcm_writel(mma, 0x00000080, MN2WS_A2PCMiNRstCtr(PCMIN1));
	// 7   : reset PCMIN   : negate
	pcm_writel(mma, 0x00000001, MN2WS_A2PCMiNRstCtr(PCMIN1));
	
	//AIN(input-sel)
	//31   : reset input-sel : assert
	pcm_writel(mma, 0x80000000, MN2WS_A2InAInIFCtrPBin2);
	//31   : reset input-sel : negate
	pcm_writel(mma, 0x00000000, MN2WS_A2InAInIFCtrPBin2);
	
	//A2D subsystem
	//31   : part reset : reset
	// 3   : PBin 2     : request
	pcm_writel(mma, 0x80000008, MN2WS_CDA2D_PARTRESET1);
	
	//A2D buffer
	//31   : part reset : reset
	// 7   : DMA ch 7   : request
	pcm_writel(mma, 0x80000004, MN2WS_CDA2D_PARTRESET0);
	
	
	//----- AIN(PCMIN1) settings -----
	//31   : 32bit mode   : off
	//10-11: input format : I2S
	//                      00 = right justified
	//                      01 = left justified
	//                      10 = I2S
	// 8- 9: bits         : 24bit
	// 4- 5: channels     : 2ch
	// 0- 3: freq         : 48kHz
	pcm_writel(mma, 0x00000800, MN2WS_A2PCMiNCtr1(PCMIN1));
	//15   : master/slave : slave
	//14   : master clock : 36.864MHz or 33.8688MHz(ignored)
	// 8- 9: output clock : 1/2(ignored)
	// 0   : enabled      : off
	pcm_writel(mma, 0x00000000, MN2WS_A2PCMiNCtr2(PCMIN1));
	// 0- 1: clock sel    : normal
	pcm_writel(mma, 0x00000000, MN2WS_A2PCMiNACCtr(PCMIN1));
	
	//----- AIN(input-sel) settings -----
	//31   : input-sel reset : negate
	//29   : ifpp3 mode      : off
	// 8-11: input-sel       : PCMIN1
	// 4- 5: endian          : 3 - 2 - 1 - 0(pass through)
	// 0- 3: format          : 2ch D0(L/R) port
	pcm_writel(mma, 0x00000000, MN2WS_A2InAInIFCtrPBin2);
	
	
	//----- A2D subsystem settings -----
	// 0   : size          : not specified(= infinite)
	pcm_writel(mma, 0x00000001, MN2WS_CDA2D_CHnCTRL1(MAIN_IN));
	//16-17: limitation id : 0
	// 0- 2: priority      : normal
	pcm_writel(mma, 0x00000005, MN2WS_CDA2D_CHnCTRL2(MAIN_IN));
	
	//16-17: endian     : 32bit big
	// 8-11: resource   : 0
	pcm_writel(mma, 0x00010000, MN2WS_CDA2D_CHnSRCAMODE(MAIN_IN));
	//16-17: endian     : 32bit big
	// 8-11: resource   : 7
	pcm_writel(mma, 0x00010700, MN2WS_CDA2D_CHnDSTAMODE(MAIN_IN));
	
	// 0-31: address    : not use
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_CHnSRCSTRTADRS(MAIN_IN));
	// 0-31: address    : not use
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_CHnDSTSTRTADRS(MAIN_IN));
	// 0-19: size       : not use
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_CHnSIZE(MAIN_IN));
	
	// 0-15: interrupts : disable all
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_CHnIE(MAIN_IN));
	
	//----- A2D buffer settings -----
	// 0-31: start address : 
	pcm_writel(mma, st, MN2WS_CDA2D_RBnBGNADRS(MAIN_IN));
	// 0-31: end address   : 
	pcm_writel(mma, ed, MN2WS_CDA2D_RBnENDADRS(MAIN_IN));
	// 0-24: threshold     : 0
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_RBnBTH(MAIN_IN));
	// 0-24: threshold     : 0
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_RBnRTH(MAIN_IN));
	// 0-31: read pointer : equal to start
	pcm_writel(mma, st, MN2WS_CDA2D_RBnRDPTR(MAIN_IN));
	
	// 0-24: pointer control : disable
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_RBnCNFG(MAIN_IN));
	//17   : write wrap flag : clear
	//16   : read wrap flag  : clear
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_RBnPTRCTRL(MAIN_IN));
	
	// 0-19: interrupts : disable all
	pcm_writel(mma, 0x00000000, MN2WS_CDA2D_RBnIE(MAIN_IN));
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0220_pcm_cap_start(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mma = &dev->map_regs[MN2WS_RES_AIO2];
	
	DPRINTF("start the capture DMA %d.\n", dev->ch);
	
	//----- AIN start -----
	// 0   : enabled    : on
	pcm_writel(mma, 0x00000001, MN2WS_A2PCMiNCtr2(PCMIN1));
	
	//----- A2D DMA start -----
	//31   : start/stop : start
	// 7   : DMA ch7    : request
	pcm_writel(mma, 0x00000080, MN2WS_CDA2D_STRT0);
	
	return 0;
}

/**
 * ꡼׶ػ
 */
static int mn2ws0220_pcm_cap_stop(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mma = &dev->map_regs[MN2WS_RES_AIO2];
	
	DPRINTF("stop the capture DMA %d.\n", dev->ch);
	
	//----- AIN start -----
	// 0   : enabled    : off
	pcm_writel(mma, 0x00000000, MN2WS_A2PCMiNCtr2(PCMIN1));
	
	//----- A2D DMA start -----
	//31   : start/stop : stop
	// 7   : DMA ch7    : request
	pcm_writel(mma, 0x80000080, MN2WS_CDA2D_STRT0);
	
	return 0;
}

/**
 * PCM Ͽ HW 񤭹ǤХåեˤĤơ
 * ߤ HW ν񤭹֤߰롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @return HW θߤν񤭹߰
 */
static unsigned long mn2ws0220_pcm_cap_get_hwptr(struct mn2ws_pcm_dev *dev)
{
	struct mem_mapping *mma = &dev->map_regs[MN2WS_RES_AIO2];
	
	return pcm_readl(mma, MN2WS_CDA2D_RBnWRPTR(MAIN_IN))
		- pcm_readl(mma, MN2WS_CDA2D_RBnBGNADRS(MAIN_IN));
}

/**
 * PCM Ͽ HW 񤭹ǤХåեˤĤơ
 * ߤΥǥХɤ߼֤ꤹ롣
 * 
 * ХåեƬ 0 Ȥ롣
 * 
 * @param dev PCM ǥХ
 * @param ptr ǥХθߤɤ߼
 */
static void mn2ws0220_pcm_cap_set_devptr(struct mn2ws_pcm_dev *dev, unsigned long ptr)
{
	//ɤ߼֤ΤפΤᡢ⤷ʤ
}

static int __init mn2ws0220_pcm_init_module(void)
{
	return mn2ws_pcm_init_module(mn2ws_pcm);
}

static void __exit mn2ws0220_pcm_exit_module(void)
{
	mn2ws_pcm_exit_module(mn2ws_pcm);
}

module_init(mn2ws0220_pcm_init_module);
module_exit(mn2ws0220_pcm_exit_module);
